pro comps_cal


;  procedure to perform comps polarimetric calibration
;  calibration files can be read in any order and the observations will be ordered

common calibrate,i_obs,wave

openpost,dev,default='s'
ans=' '

nx=2560 & ny=2160

debug='no'          				;debug mode, 'yes' or 'no'
read_in='no'

in_disk='C:\Data' & out_disk='C:\Data'
out_disk=in_disk

date='20110311'
infiles='cal_files.txt'    ;file with list of file names
indir=in_disk+'\'+date+'\'
outdir=out_disk+'\'+date+' Reduced\'
openw,2,'cal.txt'  ;open log file


;  define subarray region and region outside it

nsub=600		;size of subarray over which to average
x=rebin(findgen(nx),nx,ny)-float(nx)/2.
y=transpose(rebin(findgen(ny),ny,nx))-float(ny)/2.
subarray=where( abs(x) lt nsub and abs(y) lt nsub, complement=outside)

;  read in data or restore?

if read_in eq 'yes' then begin

;  open file containing filenames, read in filenames and associated wavelength regions

openr,1,outdir+infiles
str=' '
nfile=0
files=strarr(200)
reg=fltarr(200)
while not eof(1) do begin
	readf,1,format='(a32,f)',str,r
	files[nfile]=str
	reg[nfile]=r
	nfile=nfile+1
endwhile
close,1

files=files[0:nfile-1]
reg=reg[0:nfile-1]

regions=reg(uniq(reg,sort(reg)))			;identify wavelength regions
nreg=n_elements(regions)		;number of wavelength regions
print,nfile,' Calibration files in ',nreg,' regions'
if debug eq 'yes' then print,regions

;  loop over files and compute observed intensities, place into obs(4,7,nreg)
;  in the order of state, for each wavelength region

obs=fltarr(4,7,nreg)
state=['CLEAR','I+Q','I-Q','I+U','I-U','I+V','I-V']		;observations should put in this order

for ireg=0,nreg-1 do begin
print,regions[ireg]

	good=where(reg eq regions[ireg])		;locate files within this region

	for ical=0,6 do begin

		name=indir+files[good[ical]]
		print,name
		fits_open,name,fcb       ;open input fits file
		num=fcb.nextend
		if debug eq 'yes' then print,num,' images in file'


;  take inventory of images in this file

		comps_inventory,fcb,beam,group,wave,mod_state,type,filter,expose,diffuser,cal_state,camera
		istate=where(state eq cal_state)	;find index of cal state


;  read header of first image and get time and exposure time

		fits_read,fcb,d,header,/header_only,exten_no=1
		date_str=sxpar(header,'DATE_OBS')
		time_str=sxpar(header,'TIME_OBS')
		exposure=sxpar(header,'EXPOSURE')

		hours=float(strmid(time_str,0,2))		;  interpret time
		mins=float(strmid(time_str,2,2))
		secs=float(strmid(time_str,4,6))

		time=hours+mins/60.+secs/3600.
		if debug eq 'yes' then print,time_str,hours,mins,secs,time


;  interpolate dark image for time of observation

		dark=dark_interp(outdir,time,exposure)
		if debug eq 'yes' then print,'read dark image at:',time,' with exposure of:',exposure


;  read in and average intensity over a subarray

		for i=0,num-1 do begin
	    	fits_read,fcb,dat,header,exten_no=i+1
			dat=float(dat)-dark

			imod=i mod 4
			obs[imod,istate,ireg]=obs[imod,istate,ireg]+mean(dat[subarray])-median(dat[outside])
		endfor

		fits_close,fcb

		obs[*,istate,ireg]=4.*obs[*,istate,ireg]/float(num)
		wait,0.01
	endfor

endfor

save,regions,obs

endif else restore


;  compute fit

nreg=n_elements(regions)

ptrans=fltarr(nreg)
rtrans=fltarr(nreg)
delta=fltarr(nreg)
stokes=fltarr(4,nreg)
o_matrix=fltarr(4,4,nreg)
d_matrix=fltarr(4,4,nreg)
chi=fltarr(nreg)
e=fltarr(4,nreg)

for ireg=0,nreg-1 do begin

	i_obs=obs[*,*,ireg]
	wave=regions[ireg]
	a=comps_fit_for_omx()
	model=call_function('fit_funct',a)

	ptrans[ireg]=a[0]
	rtrans[ireg]=a[1]
	delta[ireg]=a[2]
	stokes[*,ireg]=a[3:6]
	o_mx=[[1.,a(7:9)],[1.,a(10:12)],[1.,a(13:15)],[1.,a(16:18)]]
	o_matrix[*,*,ireg]=o_mx

	a_mx=transpose(o_mx)##o_mx             ;compute A matrix
	a_invert=invert(a_mx,status,/double)       ;compute inverse of A

	eff=fltarr(4)
	for i=0,3 do eff[i]=1./sqrt(4.*a_invert(i,i))   ;compute efficiencies
	e[*,ireg]=eff

	d_mx=a_invert##transpose(o_mx)			;compute demodulation matrix
	d_matrix[*,*,ireg]=d_mx

	print,regions[ireg]
	printf,2,regions[ireg]

	print,'Polarizer trans, Retarder trans, Retardation:'
	print,a(0:2)
	printf,2,'Polarizer trans, Retarder trans, Retardation:'
	printf,2,a(0:2)

	print,'Input Stokes Vector:'
	print,a(3:6)
	printf,2,'Input Stokes Vector:'
	printf,2,a(3:6)

	print,'O Matrix:'
	print,o_mx
	printf,2,'O Matrix:'
	printf,2,o_mx

	print,'D Matrix:'
	print,d_mx
	printf,2,'D Matrix:'
	printf,2,d_mx

	print,'Observed Intensities'
	print,i_obs
	print,'Model Intensities'
	print,model
	printf,2,'Observed Intensities'
	printf,2,i_obs
	printf,2,'Model Intensities'
	printf,2,model

	chi[ireg]=call_function('min_funct',a)
	print,'Chisq: ',chi[ireg]
	printf,2,'Chisq: ',chi[ireg]

	print,'Efficiencies: ',eff
	printf,2,'Efficiencies: ',eff

endfor

if dev ne 'p' then window,0
!p.multi=[0,2,2,0,0]
plot,regions,ptrans,xtit='Wavelength (nm)',ytit='Polarizer Transmission',chars=1
oplot,regions,ptrans,psym=1
plot,regions,rtrans,xtit='Wavelength (nm)',ytit='Retarder Transmission',chars=1
oplot,regions,rtrans,psym=1
plot,regions,delta,xtit='Wavelength (nm)',ytit='Calibration Retardance',chars=1
oplot,regions,delta,psym=1
plot,regions,chi/stokes[0,*]^2,xtit='Wavelength (nm)',ytit='Chi/I^2',chars=1
oplot,regions,chi/stokes[0,*]^2,psym=1

if dev ne 'p' then window,1
yt=['Input I','Input Q/I','Input U/I','Input V/I']
plot,regions,stokes[0,*],xtit='Wavelength (nm)',ytit=yt[0]
oplot,regions,stokes[0,*],psym=1
for i=1,3 do begin
	plot,regions,stokes[i,*]/stokes[0,*],xtit='Wavelength (nm)',ytit=yt[i]
	oplot,regions,stokes[i,*]/stokes[0,*],psym=1
endfor

if dev ne 'p' then window,2
s=['I','Q','U','V']
for i=0,3 do begin
	plot,regions,e[i,*],xtit='Wavelength (nm)',ytit='Efficiency',yr=[0,1],tit=s[i]
	oplot,regions,e[i,*],psym=1
endfor

save,regions,o_matrix,d_matrix,file=date+'_cal.sav'

!p.multi=0

close,2
closepost,dev

print,'done'
end