;+
;  Name: SCD_Compute_Ematrix
;  
;  Description:
;    Procedure to compute the E matrix from the Calibration matrix (C
;    matrix) based on the formalism of deWijn described in "The
;    Concept of Optimal Calibration". See SCD Calibration writeup for 
;    details and references
;   
;  Inputs:
;    Cmatrix - calibration matrix (4 x m) comprised of the m Stokes
;              vectors produced by the calibration 
;              optics where m is the number of calibration states 
;   
;  Outputs:
;    Ematrix - E matrix (m x 4) is the nominal inverse of C as described by deWijn
;
;  Keywords:
;    cal_eff - calibration efficiency (optional, m vector), if present
;              this routine will return the calibration efficiency
;    
;  Examples:
;    SCD_Compute_Ematrix,Cmatrix,Ematrix
;    SCD_Compute_Ematrix,Cmatrix,Ematrix,cal_eff=cal_eff
;    
;  Author: S. Tomczyk
;
;  Modification History:
;
;  2022/12/17 - J. Rybak: - debug 'yes' printing of the MX, MX_invert,
;                           E matrices 
;
;-
pro SCD_Compute_Ematrix,Cmatrix,Ematrix,cal_eff=cal_eff

debug = 'yes'     ;debug mode ('yes' or 'no')
  
mx = Cmatrix##transpose(Cmatrix)

mx_invert = invert(mx,status,/double)

Ematrix = transpose(Cmatrix)##mx_invert

if debug eq 'yes' then begin

   help,Cmatrix,mx,mx_invert,Ematrix

   print
   print,' MX_matrix'
   print
   for i=0,5 do print,format='(6f10.6)',mx[*,i]
   
   print
   print,' Invert status = ',status
   
   print
   print,' MX_invert_matrix'
   print
   for i=0,5 do print,format='(6e16.8)',mx_invert[*,i]

   print
   print,' E_matrix'
   print
   for i=0,3 do print,format='(6e16.8)',Ematrix[*,i]
   print

endif

if keyword_set(cal_eff) then begin
  s = size(mx)
  n = s[1]
  cal_eff = dblarr(n)
  for i=0,s[1]-1 do cal_eff[i] = 1./sqrt(n*mx[i,i])
endif

if debug eq 'yes' then begin
   print
   if keyword_set(cal_eff) then begin
      print,' Cal_eff'
      print
      for s=0,n-1 do print,format='(6f10.6)',cal_eff[s]
      print
   endif
endif

stop

return
end
